import sys
from ctypes import *
import os

# Load the LPS Linux shared library
vnx = cdll.LoadLibrary(os.path.join(os.getcwd(), 'VNX_lps64.dll'))

# API constant: Array size for device list
MAX_LPS_DEVICES = 64

# Set API to use actual hardware
vnx.fnLPS_SetTestMode(False)

# Prepare device ID array
DeviceIDArray = c_int * MAX_LPS_DEVICES
Devices = DeviceIDArray()

# Detect devices
num_devices = vnx.fnLPS_GetNumDevices()
print(f'{num_devices} device(s) found')

if num_devices == 0:
    print("No Vaunix LPS devices located. Exiting.")
    sys.exit(1)

# Populate device ID array
dev_info = vnx.fnLPS_GetDevInfo(Devices)
print('GetDevInfo returned', dev_info)

# Example: Print model/serial, initialize, query/set frequency and phase, then close each device
for idx in range(dev_info):
    dev_id = Devices[idx]
    print(f'\n---------- Device {dev_id} ---------------------------')


    # Get model name
    ModelNameArray = create_string_buffer(64)
    model_len = vnx.fnLPS_GetModelNameA(dev_id, ModelNameArray)
    print(f'  Model: {ModelNameArray.value.decode()}')

    # Get serial number
    serial = vnx.fnLPS_GetSerialNumber(dev_id)
    print(f'  Serial number: {serial}')

    # Initialize device
    init_status = vnx.fnLPS_InitDevice(dev_id)
    print(f'  InitDevice returned {init_status}')

    # Query frequency range
    min_freq = vnx.fnLPS_GetMinWorkingFrequency(dev_id)
    max_freq = vnx.fnLPS_GetMaxWorkingFrequency(dev_id)
    print(f'  Min frequency (100kHz units): {min_freq},  Max frequency: {max_freq}')

    # Query phase shift range
    min_phase = vnx.fnLPS_GetMinPhaseShift(dev_id)
    max_phase = vnx.fnLPS_GetMaxPhaseShift(dev_id)
    print(f'  Min phase: {min_phase},  Max phase: {max_phase}')

    num_channels = vnx.fnLPS_GetNumChannels(dev_id)
    for ch in range(num_channels):
        ch += 1
        print(f'-------- Channel {ch}/{num_channels} ----------')

        vnx.fnLPS_SetChannel(dev_id, ch)

        # Read back settings
        set_freq = vnx.fnLPS_GetWorkingFrequency(dev_id)
        set_phase = vnx.fnLPS_GetPhaseAngle(dev_id)
        print(f'  Current Frequency: {set_freq}, Current Phase: {set_phase}')

        # Set frequency (example: set to midpoint of range)
        freq_mid = int((min_freq + max_freq) // 2)
        status = vnx.fnLPS_SetWorkingFrequency(dev_id, freq_mid)
        print(f'  SetWorkingFrequency({freq_mid}) returned {status}')

        # Set phase (example: set to 90 degrees)
        status = vnx.fnLPS_SetPhaseAngle(dev_id, 90)
        print(f'  SetPhaseAngle(90) returned {status}')

        # Read back settings
        set_freq = vnx.fnLPS_GetWorkingFrequency(dev_id)
        set_phase = vnx.fnLPS_GetPhaseAngle(dev_id)
        print(f'  Current Frequency: {set_freq}, Current Phase: {set_phase}')

    # Always close when done
    close_status = vnx.fnLPS_CloseDevice(dev_id)
    print(f'CloseDevice returned {close_status}')